// $Id$

/**
 * Strip whitespace from the beginning and end of a string.
 *
 * @returns {String} The trimmed string.
 */
String.prototype.trim = function () {
  return this.replace(/^ +/, '').replace(/ +$/, '');
};

/**
 * A utility method for escaping HTML tag characters.
 * <p>From Ruby on Rails.</p>
 *
 * @returns {String} The escaped string.
 */
String.prototype.htmlEscape = function () {
  return this.replace(/"/g, '&quot;').replace(/>/g, '&gt;').replace(/</g, '&lt;').replace(/&/g, '&amp;');
};

/**
 * Escapes the string without affecting existing escaped entities.
 * <p>From Ruby on Rails.</p>
 *
 * @returns {String} The escaped string
 */
String.prototype.escapeOnce = function () {
  return this.replace(/"/g, '&quot;').replace(/>/g, '&gt;').replace(/</g, '&lt;').replace(/&(?!([a-zA-Z]+|#\d+);)/g, '&amp;');
};

/**
 * <p>From Ruby on Rails.</p>
 *
 * @see http://www.w3.org/TR/html4/types.html#type-name
 */
String.prototype.sanitizeToId = function () {
  return this.replace(/\]/g, '').replace(/[^-a-zA-Z0-9:.]/g, '_');
};

/**
 * Does the string end with the specified <tt>suffix</tt>?
 * <p>From Ruby on Rails.</p>
 *
 * @param {String} suffix The specified suffix.
 * @returns {Boolean}
 */
String.prototype.endsWith = function (suffix) {
  return this.substring(this.length - suffix.length) == suffix;
};

/**
 * Does the string start with the specified <tt>prefix</tt>?
 * <p>From Ruby on Rails.</p>
 *
 * @param {String} prefix The speficied prefix.
 * @returns {Boolean}
 */
String.prototype.startsWith = function (prefix) {
  return this.substring(0, prefix.length) == prefix;
};

/**
 * Equivalent to PHP's two-argument version of strtr.
 *
 * @see http://php.net/manual/en/function.strtr.php
 * @param {Object} replacePairs An associative array in the form: {'from': 'to'}
 * @returns {String} A translated copy of the string.
 */
String.prototype.strtr = function (replacePairs) {
  var str = this;
  for (var from in replacePairs) {
    str = str.replace(new RegExp(from, 'g'), replacePairs[from]);
  }
  return str;
};
